// déclaration des fonctions externes
#include <stdio.h>  // nécessaire pour printf
#include <stdlib.h> // nécessaire pour malloc et free
#include <string.h> // pour strdup

// DEFINITION D'UN TYPE BOOLEEN
#include <stdbool.h> // type booléen (bool: true/false)

// Définit un type Element, comme char*
typedef char* Element;

// la constante element_invalide
const static Element element_invalide = (Element)NULL;

// - les opérations (fonctions):

// Vérifie la validité d'un élément
// @param E: Element à vérifier
// @return true si E est valide, faux sinon
bool ElementEstValide(Element E) 
{
	return E != element_invalide ? true : false;
}

// Affiche un élément
// @param E: élément à afficher
// @Note affiche <INVALIDE> en cas d'élément invalide
void ElementAfficher(Element E)
{
	printf("%s\n", ElementEstValide(E) ? E : "<INVALIDE>");
}

// Compare deux éléments
// @param E1, E2: les deux éléments à comparer
// @return true si E1 et E2 sont égaux, faux sinon
// @Note si E1 et E2 sont tous les deux invalides, la fonction renvoie true
bool ElementComparer(Element E1, Element E2)
{
	// strcmp requiert que les deux pointeurs soient non nuls
	if (ElementEstValide(E1) && ElementEstValide(E2)) return strcmp(E1,E2) ? false : true;
	return E1 == E2; // E1 ou E2 est NULL, donc la comparaison peut se faire par pointeurs
}

// Effectue une copie profonde d'un élément (donc allocation de mémoire)
// @param E: élément à copier
// @return nouvel élément avec le même contenu que E. 
//         Renvoie element_invalide si E n'est pas valide
// @Note l'élément renvoyé peut être détruit par appel à la fonction free.
//       Voir aussi la fonction ElementDetruire qui le fait proprement
Element ElementCopie(Element E)
{
	if (ElementEstValide(E)) return strdup(E);
	return E;
}

// Destruction d'un élement (libération de mémoire)
// @param E: élément à détruire
// @return element_invalide
// @Note fonctionne même si E est invalide
Element ElementDetruire(Element E)
{
	if (ElementEstValide(E)) free(E);
	return element_invalide;
}


// DEFINITION DES SORTES POUR LA LISTE CHAINEE
// définition d'une cellule
struct cellule {
  Element data;
  struct cellule *next;
};
typedef struct cellule cellule;

// définition du type Liste
typedef cellule *Liste;

// définition de la constante liste_vide
const static Liste liste_vide = (Liste)NULL;

// OPERATIONS SUR LA SORTE
// Vérifie si une liste est vide
// @param L : la liste à évaluer
// @return true si la liste est vide, false sinon
bool EstVide(Liste L) { return L == liste_vide ? true : false; }

// Renvoie l'élément stocké en tête de liste (accès en lecture)
// @param L : la Liste dont on veut explorer la tête
// @return l'élément stocké (copie superficielle)
// @Note renvoie element_invalide si la liste est vide
Element ContenuLire(Liste L) {
  return EstVide(L) ? element_invalide : L->data;
}

// Stocke un élément en tête de liste (copie profonde)
// @param L: la liste dont on veut modifier la tête
//        E: l'élément à stocker en tête de liste
// @Note ne fait rien si la liste est vide ou si l'élément est invalide
void ContenuModif(Liste L, Element E) {
  if (!EstVide(L) && ElementEstValide(E)) {
		ElementDetruire(L->data); // un élément est déjà stocké. Il a été inséré par copie profonde, donc il faut d'abord libérer la mémoire associée
    L->data = ElementCopie(E); // copie profonde du nouvel élément
	}
}

// Renvoie le successeur de la première cellule, autrement dit la liste qui
// commence à la deuxième cellule
// @param L: la liste à interroger
// @return son successeur
// @Note renvoie liste_vide si L est vide
Liste SuccLire(Liste L) {
  return EstVide(L) ? liste_vide : L->next;
}

// Modifie le successeur de la première cellule d'une liste (copie superficielle)
// @param L: la liste à modifier
//        N: la liste à insérer
// @Note usage typique quand L ne contient qu'une seule cellule (L->next est vide)
//       la fonction insère alors cette cellule à la suite de L, autrement dit L->next==N
//       en sortie de la fonction
//       Cette fonction ne fait rien si L est vide
void SuccModif(Liste L, Liste N) {
  if (!EstVide(L))
    L->next = N;
}

// Création d'une liste avec les informations d'un élement et d'une liste comme successeur
// @param E: l'élément à stocker dans la première cellule
//        L: la liste qui suivra la première cellule
// @return la nouvelle liste, dont la première cellule contient l'élément E et a pour 
//         successeur L
// @Note : renvoie L si E est invalide (aucune cellule n'est ajoutée)
Liste Creer(Element E, Liste L) {
  if (!ElementEstValide(E))
    return L;

	// calloc fait comme un malloc mais initialise
	// toute la zone allouée à 0. Utile pour ContenuModif
	// qui fait appel à ElementDetruire
	// qui nécessite que l'Element stocké soit bien
	// initialisé à 0 s'il n'est pas alloué
  Liste ret = calloc(1,sizeof(cellule));
	ContenuModif(ret,E);
	SuccModif(ret,L);
  return ret;
}

// Détruit la première cellule d'une liste (libération mémoire)
// @param L: liste dont on veut détruire la première cellule
// @return la liste commençant à la deuxième cellule de L
// @Note renvoie liste_vide si L est vide
Liste Detruire(Liste L) {
  if (EstVide(L))
    return liste_vide;
	ElementDetruire(ContenuLire(L));
  Liste tmp = SuccLire(L);
  free(L);
  return tmp;
}

int main(int argc, char *argv[])
{
// création et affichage

  int i;
  Liste L = liste_vide;
  for (i = 1; i < argc; i++)
    L = Creer(argv[i], L); // gère le cas où E est invalide: rien n'est ajouté à la liste

// pour le TP sur les listes: exemples d'appel des fonctions
// vues en TD
//  // test de Longueur
//  printf("La longueur de la liste est : %d\n", Longueur(L));
//
// // Afficher
//  printf("Affichage de la liste\n");
//  Afficher(L);
        return 0;
}

