function [ Cartoon , Texture ] = NoLoDuDoCT( Ima1, coords, size_patches, param_CT, std_gray, nbest, stepE, stepA )
% code for:
% "A Non-Local Dual-Domain Approach to Cartoon and Texture Decomposition"
% Frederic Sur
% IEEE Transactions on Image Processing
% 2019
% usage:
% Ima1: image to be decomposed
% coords: array of coordinates for patch visualization, default [] (see script_NoLoDuDoCT.m) 
% size_patches: size of the patches, default 32
% param_CT: parameter sigma_{CT} for the pre-processing step described in Buades et al. fast cartoon + texture decomposition, default 2
% std_gray: param beta in the paper, default 20
% nbest: number of best image patches considered to estimate statistics, default 20
% stepE: subsampling factor to extract patches for estimating statistics (cf {\cal D}_s in the paper), default size_patches/4
% stepA: subsampling factor to build cartoon and texture decomposition in the inverse windowed Fourier transform, default size_patches/8

disp(' ')
disp('NoLoDuDoCT v. 1.0')
disp('Please refer to:')
disp('A Non-Local Dual-Domain Approach to Cartoon and Texture Decomposition')
disp('Frederic Sur')
disp('IEEE Transactions on Image Processing')
disp('2019.')
disp(' ')
drawnow('update')


switch nargin
    case 1
        coords=[];
        size_patches=32;
        param_CT=2;
        std_gray=20;
        nbest=20;
        stepE=round(size_patches/4);
        stepA=round(size_patches/8);
    case 2
        size_patches=32;
        param_CT=2;
        std_gray=20;
        nbest=20;
        stepE=round(size_patches/4);
        stepA=round(size_patches/8); 
    case 3
        param_CT=2;
        std_gray=20;
        nbest=20;
        stepE=round(size_patches/4);
        stepA=round(size_patches/8); 
    case 4
        std_gray=20;
        nbest=20;
        stepE=round(size_patches/4);
        stepA=round(size_patches/8);  
    case 5
        nbest=20;
        stepE=round(size_patches/4);
        stepA=round(size_patches/8);   
    case 6
        stepE=round(size_patches/4);
        stepA=round(size_patches/8);
end

% in color images, texture components are detected in the luminance channel
if (size(Ima1,3)==3)
    Ima=double(rgb2gray(uint8(Ima1))); 
else
    Ima=Ima1; 
end

% padding to avoid border effects
Ima=padarray(Ima,[size_patches/2,size_patches/2],'symmetric','both');
Ima1=padarray(Ima1,[size_patches/2,size_patches/2],'symmetric','both');
[rows, cols] = size(Ima);
coords=[coords+size_patches/2, ones(size(coords,1),1)];

tic

% Tentative rough decomposition  (Buades et al.'s "Fast cartoon + texture image filters")
[C,T]=fastcartoontexture(Ima,param_CT);

% total number of patches for estimation
P=numel((1+size_patches/2):stepE:(rows-size_patches/2))*numel((1+size_patches/2):stepE:(cols-size_patches/2));

% Gaussian window in WFT
[X,Y]=meshgrid(0:size_patches-1,0:size_patches-1);
H=exp((-(X-size_patches/2).^2-(Y-size_patches/2).^2)/(size_patches/5)^2/2);
H=H/max(H(:));

% LowFreq: mask marking low frequencies
[fx, fy] = meshgrid((-size_patches/2:size_patches/2-1)/size_patches, (-size_patches/2:size_patches/2-1)/size_patches);
Freq=sqrt(fx.^2+fy.^2); 
f2=2/size_patches; 
LowFreq=ifftshift(Freq<f2);

% build patches
fftPatches=zeros(size_patches*size_patches,P);  % array of the fft of each patch
fftPatchesC=zeros(size_patches*size_patches,P);  % array of the fft of the cartoon component of each patch
PatchesC=zeros(size_patches*size_patches,P);  % array of the cartoon component of each patch
Kernels=zeros(size_patches*size_patches,P);  % array of the kernel for each patch (g_\alpha*g_\beta in the paper)
list_coords=zeros(size_patches*size_patches,2);  % array of the coordinates of each patch
n=0;
for i=(1+size_patches/2):stepE:(rows-size_patches/2)
  for j=(1+size_patches/2):stepE:(cols-size_patches/2)
    n = n+1;
    PatchC = C(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1);
    K = H .* exp( -( C(i,j) - PatchC ).^2/ (2*std_gray^2) );% / (2*pi*std_gray^2);
    imagette = K.*Ima(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1)/256;
    imagetteC = K.*C(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1)/256;
    fftimagette = fft2(imagette);
    fftimagette(LowFreq) = 0;
    fftimagetteC = fft2(imagetteC);
    fftPatches(:,n) = fftimagette(:);
    fftPatchesC(:,n) = fftimagetteC(:);
    PatchesC(:,n) = K(:).*PatchC(:)/256;
    Kernels(:,n) = K(:);
    list_coords(n,:)=[i,j];
  end
end

% power spectra:
powspecPatches=abs(fftPatches).^2;
powspecPatchesC=abs(fftPatchesC).^2;

% FLANN  https://www.cs.ubc.ca/research/flann/
build_params.algorithm='kdtree';
build_params.trees=1;
[ann_index,ann_parameters,ann_speedup]=flann_build_index(powspecPatches,build_params);


% pixelwise decomposition

n=0;
newIma2=zeros(size(Ima1));
normIma2=zeros(size(Ima));
Niter=numel((1+size_patches/2):stepA:(rows-size_patches/2))*numel((1+size_patches/2):stepA:(cols-size_patches/2));
ndisp=1;
writepdf=false;  % optional: write output images in pdf files
rootfilename='name';  % change filename

for i=(1+size_patches/2):stepA:(rows-size_patches/2)  
    for j=(1+size_patches/2):stepA:(cols-size_patches/2)
        n=n+1;
        if (mod(n,1000)==1), disp([num2str(n),' / ',num2str(Niter)]); end
        PatchC = C(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1);
        K = H .* exp( -( C(i,j) - PatchC ).^2/ (2*std_gray^2) );% / (2*pi*std_gray^2); 
        imagette=K.*Ima(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1)/256;
        fftPatch = fft2(imagette);
        fftPatch(LowFreq) = 0;
        powspecPatch = abs(fftPatch(:)).^2;
        
        % search the nbest nearest patches with flann and calculate the
        % weights in coefdistpatch (w_i in the paper)
        [inddistpatch,distpatch] = flann_search(ann_index,powspecPatch,nbest,ann_parameters);  
        lambda=quantile(distpatch,.5);
        coefdistpatch=exp(-distpatch'.^2/lambda^2);
        Norm=sum(coefdistpatch);
        if (lambda==0) 
            coefdistpatch=ones(size(distpatch'))/nbest;
        else
            coefdistpatch=coefdistpatch/Norm;
        end   
        
        % Eq. 14
        meanpowspecPatchesC = sum( repmat(coefdistpatch,size_patches*size_patches,1) .* powspecPatchesC(:,inddistpatch) , 2);
        % Eq. 15
        varpowspecPatchesC = nbest/(nbest-1) * sum( repmat(coefdistpatch,size_patches*size_patches,1) .* ...
            ( powspecPatchesC(:,inddistpatch) - repmat(meanpowspecPatchesC,1,nbest) ).^2 ,2) ;

        % Eq. 17
        meanpowspecPatches = sum( repmat(coefdistpatch,size_patches*size_patches,1) .* powspecPatches(:,inddistpatch) , 2);      
        I=find(fftshift(Freq)>=0.5);
        data=meanpowspecPatches(I,:)-meanpowspecPatchesC(I,:);
        varnoise=mean(data(:));
        
        % probability according to Eq. 20
        proba=normcdf(powspecPatch,meanpowspecPatchesC+varnoise,sqrt(varpowspecPatchesC+2*varnoise^2+4*meanpowspecPatchesC*varnoise),'upper');
        proba(isnan(proba))=1;    
        
        % Mask indicating periodic texture components in the frequency domain 
        % (1: no texture, 0:texture)
        Mask=(proba>.05/(size_patches*size_patches))|(LowFreq(:));
        
        % optional display
        if ~isempty(coords)
           inddisp=(coords(:,3)==1)&(coords(:,1)<i)&(coords(:,2)<j);
           if any(inddisp)
                filename=[rootfilename,'_patch',num2str(ndisp)];   
                ndisp=ndisp+1;
                maxi=max(log(powspecPatch));
                mini=min(log(powspecPatch(log(powspecPatch)~=-Inf)));
                scale=[mini+.4*(maxi-mini),maxi-.1*(maxi-mini)];
                figure, imagesc(K.*Ima(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1),[0,255]), colormap(gray), colorbar;
                title('Image patch');
                if (writepdf) print(filename,'-dpdf'); end
                figure, imagesc(fftshift(reshape(log(meanpowspecPatchesC+varnoise),size_patches,size_patches)),scale); colorbar; 
                title('Sample expectation');
                if (writepdf) print([filename,'_sampleExp'],'-dpdf'); end
                figure, imagesc(fftshift(reshape(log(sqrt(varpowspecPatchesC+2*varnoise^2+4*meanpowspecPatchesC*varnoise)),size_patches,size_patches)),scale); colorbar; 
                title('Sample standard deviation');
                if (writepdf) print([filename,'_sampleStd'],'-dpdf'); end
                figure, imagesc(fftshift(reshape(log(powspecPatch),size_patches,size_patches)),scale); colorbar; 
                title('Patch power spectrum');
                if (writepdf) print([filename,'_powspec'],'-dpdf'); end
                figure, imagesc(fftshift(reshape(proba,size_patches,size_patches)),[0,1]); colorbar;
                title('Probability map');
                if (writepdf) print([filename,'_proba'],'-dpdf'); end
                figure, imagesc(fftshift(reshape(1-Mask,size_patches,size_patches)),[0,1]); colorbar; 
                title('Mask');
                if (writepdf) print([filename,'_Mask'],'-dpdf'); end
                figure, imagesc(Ima), colormap(gray), hold on;
                for c=1:nbest,
                    rectangle('Position',[list_coords(inddistpatch(c),2)-size_patches/2,list_coords(inddistpatch(c),1)-size_patches/2,size_patches,size_patches],'EdgeColor','g');
                    strweight = num2str(coefdistpatch(c),2);
                    text(list_coords(inddistpatch(c),2)-size_patches/2,list_coords(inddistpatch(c),1)+size_patches/2,strweight,'Color','white');
                end
                rectangle('Position',[j-size_patches/2,i-size_patches/2,size_patches,size_patches],'EdgeColor','r');
                title('Position of the similar patches');
                if (writepdf) print([filename,'_pos'],'-dpdf'); end
                coords(find(inddisp),3)=0; 
           end
       end
       
       % building the cartoon component by taking inverse Fourier transform after masking texture component in windowed Fourier transform
       % (line 16 in Algorithm 1)
       if (size(Ima1,3)==3)
            fftPatch=fft2(repmat(H,1,1,3).*Ima1(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1,:)).*repmat(reshape(Mask,size_patches,size_patches),1,1,3);
            newPatch=ifft2(fftPatch,'symmetric');
       else
            fftPatch=fft2(H.*Ima(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1)).*reshape(Mask,size_patches,size_patches);
            newPatch=ifft2(reshape(fftPatch,size_patches,size_patches),'symmetric');
       end
       newIma2(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1,:) = newIma2(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1,:) + newPatch;
       normIma2(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1) = normIma2(i-size_patches/2:i+size_patches/2-1,j-size_patches/2:j+size_patches/2-1) + H;
    
    end   
end

% normalization in the inverse Fourier transform
nH=sum(sum(H(1:stepA:end,1:stepA:end)));
normIma2=normIma2/nH;
newIma2=newIma2/nH;
Cartoon=newIma2./repmat(normIma2,1,1,size(Ima1,3));
Texture=Ima1-Cartoon;
Cartoon=Cartoon(1+size_patches/2:end-size_patches/2,1+size_patches/2:end-size_patches/2,:);
Texture=Texture(1+size_patches/2:end-size_patches/2,1+size_patches/2:end-size_patches/2,:);

toc

% display
figure, imshow(uint8(Cartoon));
figure, imshow(uint8(128+3*Texture));  % amplitude of the texture component multiplied by 3 for a better readability

return

 