/*
Copyright CNRS                                                           
contributor: Yann Guermeur 11/04/09

Yann.Guermeur@cnrs.fr

This software is a computer program whose purpose is to perform multi-category
discrimination with a multi-class support vector machine: the LLW-M-SVM.

This software is governed by the CeCILL-B license under French law and
abiding by the rules of distribution of free software.  You can  use, 
modify and/ or redistribute the software under the terms of the CeCILL-B
license as circulated by CEA, CNRS and INRIA at the following URL
"http://www.cecill.info". 

As a counterpart to the access to the source code and  rights to copy,
modify and redistribute granted by the license, users are provided only
with a limited warranty  and the software's author,  the holder of the
economic rights,  and the successive licensors  have only  limited
liability. 

In this respect, the user's attention is drawn to the risks associated
with loading,  using,  modifying and/or developing or reproducing the
software by the user in light of its specific status of free software,
that may mean  that it is complicated to manipulate,  and  that  also
therefore means  that it is reserved for developers  and  experienced
professionals having in-depth computer knowledge. Users are therefore
encouraged to load and test the software's suitability as regards their
requirements in conditions enabling the security of their systems and/or 
data to be ensured and,  more generally, to use and operate it in the 
same conditions as regards security. 

The fact that you are presently reading this means that you have had
knowledge of the [CeCILL|CeCILL-B|CeCILL-C] license and that you accept its terms.
*/

/*--------------------------------------------------------------------------*/
/*  Name           : eval_SVM.c                                             */
/*  Version        : 1.0                                                    */
/*  Creation       : 06/20/08                                               */
/*  Last update    : 10/22/25                                               */
/*  Subject        : Implementation of the LLW-M-SVM                        */
/*  Module         : M-SVM used in test                                     */
/*  Author         : Yann Guermeur Yann.Guermeur@cnrs.fr                    */
/*--------------------------------------------------------------------------*/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "algebre.h"
#include "biblio.h"

#define max(a,b) ((a)>=(b)?(a):(b))

#define true 1
#define false 0
#define taille 81
#define very_small 1e-5
#define step 100

FILE *fs, *fc;

int status;

long Q=0, i, j, k, l, dim_input, nb_data_app=0, nb_data_test=0, y_i,
ind_max, **X_app, **X_test, *y_app, *y_test,
smallest_cat=0, largest_cat=0, nb_symb, **mat_conf;

char fichier_app[taille], fichier_alpha[taille], fichier_dot_prod_aa[taille],
fichier_test[taille], commande[taille+20],
fichier_fichcom[taille], fichier_resultat[taille], fichier_vect_theta[taille],
var_open = '(', var_close = ')';

double **alpha, *b_SVM, C=0.0, *vect_theta, **dot_prod_aa,
primal=0.0, dual=0.0, Q_form_dual=0.0, **gradient, max_alpha=0.0,
**H_alpha, *partial_average_alpha, norm, partial, **xi, cost=0.0,
L_form_dual=0.0, maximum,
sum_all_alpha=0.0, kernel=0.0, alpha_H_alpha=0.0, QminusOneInv=0.0;

/* Functions included in this program */

void caract_db();
void read_data();
void alloc_memory();
void read_alpha();
void check_feasible_sol();
void read_vect_theta();
void read_dot_prod_aa();
void compute_gradient();
void estimate_b();
void eval_training();
void compute_obj_dual();
void compute_obj_primal();
double fonction(long categorie, long *vecteur);
void compute_outputs(long **X, long *y, long nb_data);

int main(int argc, char *argv[])

{

strcpy(fichier_fichcom, argv[1]);

status = system("clear");
caract_db();
read_dot_prod_aa();
read_data();
alloc_memory();
read_alpha();
read_vect_theta();

if(max_alpha > 0.0)
  {
  check_feasible_sol();
  compute_gradient();
  estimate_b();
  }

eval_training();
compute_obj_dual();
compute_obj_primal();

display_stats(fichier_app, nb_data_app, Q, mat_conf, cost);

Pause("\a=> New Line to access the test performance...");

printf("\n*** Test performance\n");
compute_outputs(X_test, y_test, nb_data_test);

display_stats(fichier_test, nb_data_test, Q, mat_conf, cost);

}

void caract_db()

{

if((fs=fopen(fichier_fichcom, "r"))==NULL)
  {
  printf("\nFile of parameters %s: cannot be open...\n", fichier_fichcom);
  exit(0);
  }

status = fscanf(fs, "%lf", &C);
status = fscanf(fs, "%s", fichier_app);
printf("\nThe file of the training set is: %s", fichier_app);
status = fscanf(fs, "%s", fichier_test);
printf("\n    The file of the test set is: %s\n", fichier_test);

status = fscanf(fs, "%s", fichier_alpha);
status = fscanf(fs, "%s", fichier_vect_theta);
status = fscanf(fs, "%s", fichier_dot_prod_aa);
status = fscanf(fs, "%s", fichier_resultat);

fclose(fs);

}

void read_data()

{

long min_y, ind_min_y, max_y, ind_max_y, *cardinal_cat, cardinal_min,
cardinal_max=0;

if((fs=fopen(fichier_app, "r"))==NULL)
  {
  printf("\nFile of data: %s cannot be open...\n", fichier_app);
  exit(0);
  }

status = fscanf(fs, "%ld", &nb_data_app);
status = fscanf(fs, "%ld", &dim_input);
status = fscanf(fs, "%ld", &Q);

QminusOneInv = 1.0 / (Q - 1.0);

mat_conf = matrix_l(Q,Q);
cardinal_cat = (long *) calloc(Q+1, sizeof(long));

X_app = matrix_l(nb_data_app, dim_input);
y_app = (long *) calloc(nb_data_app+1, sizeof(long));

min_y = nb_data_app;
max_y = 0;

for(i=1; i<=nb_data_app; i++)
  {
  for(j=1; j<=dim_input; j++)
    {
    status = fscanf(fs, "%ld", &X_app[i][j]);
    if(X_app[i][j] == 0)
      X_app[i][j] = nb_symb;
    }
  status = fscanf(fs, "%ld", &y_app[i]);
  if(y_app[i] < min_y)
    {
    min_y = y_app[i];
    ind_min_y = i;
    }
  if(y_app[i] > max_y)
    {
    max_y = y_app[i];
    ind_max_y = i;
    }
  }

fclose(fs);

if((fs=fopen(fichier_test, "r"))==NULL)
  {
  printf("\nFile of data: %s cannot be open...\n", fichier_test);
  exit(0);
  }

status = fscanf(fs, "%ld", &nb_data_test);
status = fscanf(fs, "%ld", &dim_input);
status = fscanf(fs, "%ld", &Q);

X_test = matrix_l(nb_data_test, dim_input);
y_test = (long *) calloc(nb_data_test+1, sizeof(long));

for(i=1; i<=nb_data_test; i++)
  {
  for(j=1; j<=dim_input; j++)
    {
    status = fscanf(fs, "%ld", &X_test[i][j]);
    if(X_test[i][j] == 0)
      X_test[i][j] = nb_symb;
    }
  status = fscanf(fs, "%ld", &y_test[i]);
  if(y_test[i] < min_y)
    min_y = y_test[i];
  if(y_test[i] > max_y)
    max_y = y_test[i];
  }

fclose(fs);

/*
printf("\nExtreme indices of categories : %2ld -> %2ld\n", min_y, max_y);
*/

if(((min_y != 0) && (min_y != 1)) || (max_y - min_y != Q-1))
  {
  printf("\nWrong numbering of the categories\n");
  exit(0);
  }

if(min_y == 0)
  {
  for(i=1; i<=nb_data_app; i++)
    y_app[i]++;
  for(i=1; i<=nb_data_test; i++)
    y_test[i]++;
  }

for(k=1; k<=Q; k++)
  cardinal_cat[k] = 0;

for(i=1; i<=nb_data_app; i++)
  cardinal_cat[y_app[i]]++;

cardinal_min = nb_data_app;

for(k=1; k<=Q; k++)
  {
  if(cardinal_cat[k] > cardinal_max)
    {
    cardinal_max = cardinal_cat[k];
    largest_cat = k;
    }
  if(cardinal_cat[k] < cardinal_min)
    {
    cardinal_min = cardinal_cat[k];
    smallest_cat = k;
    }
  }

/*
for(k=1; k<=Q; k++)
  printf("\nCardinal of category %2ld: %5ld", k, cardinal_cat[k]);

printf("\nIndex of the smallest category: %2ld", smallest_cat);
printf("\nIndex of the largest category: %2ld", largest_cat);
Pause("");
*/

if(cardinal_min == 0)
  {
  printf("\nNo training example for category %2ld\n", smallest_cat);
  exit(0);
  }

}

void alloc_memory()

{

alpha = matrix(nb_data_app, Q);
vect_theta = (double *) calloc(dim_input+1, sizeof(double));
b_SVM = (double *) calloc(Q+1, sizeof(double));

for(k=1; k<=Q; k++)
  b_SVM[k] = 0.0;

gradient = matrix(nb_data_app, Q);
H_alpha  = matrix(nb_data_app, Q);

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
    H_alpha[i][k] = 0.0;

for(i=1; i<=nb_data_app; i++)
  {
  y_i = y_app[i];
  for(k=1; k<=Q; k++)
    if(k != y_i)
      gradient[i][k] = QminusOneInv;
    else
      gradient[i][k] = 0.0;
  }

}

void read_alpha()

{

max_alpha = 0.0;

if((fs=fopen(fichier_alpha, "r"))==NULL)
  {
  printf("\nFile of dual variables %s: cannot be open...\n", 
  fichier_alpha);
  exit(0);
  }

partial_average_alpha = (double *) calloc(nb_data_app+1, sizeof(double));

for(i=1; i<=nb_data_app; i++)
  {
  partial_average_alpha[i] = 0.0;
  for(k=1; k<=Q; k++)
    {
    status = fscanf(fs, "%lf", &alpha[i][k]);
    sum_all_alpha += alpha[i][k];
    partial_average_alpha[i] += alpha[i][k];

    if(alpha[i][k] > max_alpha)
      max_alpha = alpha[i][k];
    }
  partial_average_alpha[i] /= Q;
  }

fclose(fs);

/* printf("\nThe maximum value among the dual variables is: %e\n", max_alpha); */

}

void check_feasible_sol()

{

double *constraints;

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
    if(((k == y_app[i]) && (alpha[i][k] != 0.0)) || (alpha[i][k] < 0.0)
       || (alpha[i][k] > C))
      {
      printf("\nNo feasible solution: alpha[%ld][%ld] = %lf\n\n",
             i, k, alpha[i][k]);
      exit(0);
      }
     
constraints = (double *) calloc(Q+1, sizeof(double));

for(k=1; k<=Q; k++)
  constraints[k] = sum_all_alpha / Q;

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
    constraints[k] -= alpha[i][k];  

printf("\nSatisfaction of the equality constraints:\n\n");

for(k=1; k<=Q; k++)
  printf("%s%e\n", constraints[k] < 0.0 ? " " : "  ", constraints[k]);

norm = 0.0;

for(k=1; k<=Q; k++)
  norm += constraints[k] * constraints[k];

norm /= (double)Q;
norm = sqrt(norm);

if(norm >= very_small)
  printf("\nLarge deviation of the equality constraints...\n");

sprintf(commande, "cp %s Feasible/.", fichier_alpha);
status = system(commande);

}

void read_vect_theta()

{

if((fs=fopen(fichier_vect_theta, "r"))==NULL)
  {
  printf("\nFile of the positional weighting %s: cannot be open...\n",
  fichier_vect_theta);
  exit(0);
  }
else
  printf("\nThe file of the positional weighting is: %s", fichier_vect_theta);

for(i=1; i<=dim_input; i++)
  status = fscanf(fs, "%lf", &vect_theta[i]);

fclose(fs);

}

void read_dot_prod_aa()

{

if((fs=fopen(fichier_dot_prod_aa, "r"))==NULL)
  {
  printf("\nFile of the substitution matrix %s: cannot be open...\n",
  fichier_dot_prod_aa);
  exit(0);
  }
else
  printf("\n The file of the substitution matrix is: %s", fichier_dot_prod_aa);

status = fscanf(fs, "%ld", &nb_symb);
dot_prod_aa = matrix(nb_symb, nb_symb);

for(i=1; i<=nb_symb; i++)
  for(j=1; j<=nb_symb; j++)
    status = fscanf(fs, "%lf", &dot_prod_aa[i][j]);

fclose(fs);

}

void compute_gradient()

{

printf("\nStart of the gradient computation...\n");

for(i=1; i<=nb_data_app; i++)
  {
  if((i%step) == 0)
    printf("\n%9ld", i);
  y_i = y_app[i];

  for(k=1; k<=Q; k++)
    {
    H_alpha[i][k] = 0.0;
    if(k != y_i)
      {
      for(j=1; j<=nb_data_app; j++)
        {
        partial = alpha[j][k] - partial_average_alpha[j];

	if(partial != 0.0)
          {
          kernel = 
          gaussian(X_app[i], X_app[j], vect_theta, dot_prod_aa, dim_input);
	  H_alpha[i][k] += partial * kernel;
          }

        }
      }
    }
  }

for(i=1; i<=nb_data_app; i++)
  {
  y_i = y_app[i];
  for(k=1; k<=Q; k++)
    if(k != y_i)
      gradient[i][k] = - H_alpha[i][k] + QminusOneInv;
    else
      gradient[i][k] = 0.0;
  }

printf("\n\nEnd of the gradient computation...\n");

}

double fonction(long categorie, long *vecteur)

{

long index;

double resultat = 0.0;

for(index=1; index<=nb_data_app; index++)
  {
  partial = partial_average_alpha[index] - alpha[index][categorie];

  if(partial != 0.0)
  resultat += partial * 
  gaussian(X_app[index], vecteur, vect_theta, dot_prod_aa, dim_input);
  }

return resultat;

}

void eval_training()

{

double *output;

output = (double *) calloc(Q+1, sizeof(double));
xi = matrix(nb_data_app, Q);

for(k=1; k<=Q; k++)
  for(l=1; l<=Q; l++)
    mat_conf[k][l] = 0;

/*
printf("\n\n*** Computing the training performance...\n");
Pause("");
*/

for(i=1; i<=nb_data_app; i++)
  {
  y_i = y_app[i];
  output[y_i] = 0.0;

  for(k=1; k<=Q; k++)
    if(k != y_i)
      {
      output[k] = -H_alpha[i][k] + b_SVM[k];
      output[y_i] -= output[k];
      xi[i][k] = max(gradient[i][k] + b_SVM[k], 0.0);
      }

  ind_max = 1;
  maximum = output[1];

  for(k=2; k<=Q; k++)
    if(output[k] > maximum)
      {
      maximum = output[k];
      ind_max = k;
      }

  if(output[largest_cat] == maximum)
    ind_max = largest_cat;

  mat_conf[y_i][ind_max]++;
  }

/*
printf("\n\n*** Training performance computed...\n");
Pause("");
*/

}

void estimate_b()

{

long *ind_sup_vect, nb_known_b = 0;

double center = C/2.0, current_dist, *dist_min, sum=0.0;

ind_sup_vect = (long *) calloc(Q+1, sizeof(long));
dist_min = (double *) calloc(Q+1, sizeof(double));

for(k=1; k<=Q; k++)
  {
  ind_sup_vect[k] = 0;
  dist_min[k] = center;
  }

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
    if((alpha[i][k] > 0.0) && (alpha[i][k] < C))
      {
      current_dist = fabs(alpha[i][k] - center);
      if(current_dist < dist_min[k])
        {
        dist_min[k] = current_dist;
	ind_sup_vect[k] = i;
        }
      }

for(k=1; k<=Q; k++)
  if(ind_sup_vect[k] > 0)
    {
    nb_known_b++;
    b_SVM[k] = -gradient[ind_sup_vect[k]][k];
    sum += b_SVM[k];
    }
  else
    printf("\nMissing SV for category %ld\n", k);

if(nb_known_b < Q)
  {
  if(nb_known_b < Q-1)
    {
    printf("\n%ld unknown biases...\n\n", Q-nb_known_b);
    exit(0);
    }
  else
    printf("\nOne unknown bias\n");
  }

if(nb_known_b == Q-1)
  for(k=1; k<=Q; k++)
    if(ind_sup_vect[k] == 0)
      {
      b_SVM[k] = -sum;
      sum = 0.0;
      }

printf("\nSum of the components of b: %e\n", sum);

if(sum != 0.0)
  {
  sum /= Q;
  for(k=1; k<=Q; k++)
    b_SVM[k] -= sum;
  }

}

void compute_outputs(long **X, long *y, long nb_data)

{

double *output;

output = (double *) calloc(Q+1, sizeof(double));

for(k=1; k<=Q; k++)
  for(l=1; l<=Q; l++)
    mat_conf[k][l] = 0;

if((fc=fopen(fichier_resultat, "w"))==NULL)
  {
  printf("\nFile of outputs %s: cannot be open...\n", fichier_resultat);
  exit(0);
  }

for(i=1; i<=nb_data; i++)
  {
  for(k=1; k<=Q; k++)
    output[k] = fonction(k, X[i]) + b_SVM[k];

  ind_max = 1;
  maximum = output[1];

  for(k=2; k<=Q; k++)
    if(output[k] > maximum)
      {
      maximum = output[k];
      ind_max = k;
      }
  
  if(output[largest_cat] == maximum)
    ind_max = largest_cat;

  mat_conf[y[i]][ind_max]++;

  for(k=1; k<=Q; k++) 
    fprintf(fc, "%12.6lf%c", output[k], (k==Q) ? '\n' : ' ');

  if((i % step) == 0)
    printf("\nExample: %5ld", i);
  }

fclose(fc);

printf("\n");

}

void compute_obj_dual()

{

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
    alpha_H_alpha += H_alpha[i][k] * alpha[i][k];

Q_form_dual = alpha_H_alpha;
L_form_dual = sum_all_alpha / (Q - 1.0);

dual = -0.5 * Q_form_dual + L_form_dual;

}

void compute_obj_primal()

{

double R_emp = 0.0;

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
    R_emp += xi[i][k];

primal = 0.5 * alpha_H_alpha + C * R_emp;

printf("\n*** Training performance\n");
printf("\n            Dual objective function: %e", dual);
printf("\nEstimated primal objective function: %e", primal);

if(primal != 0.0)
  printf("\n%cratio %5.3lf%c\n", var_open, dual / primal, var_close);

}
