/* Copyright 2008, 2009 Yann Guermeur                                       */

/* This program is free software; you can redistribute it and/or modify     */
/* it under the terms of the GNU General Public License as published by     */
/* the Free Software Foundation; either version 2 of the License, or        */
/* (at your option) any later version.                                      */

/* This program is distributed in the hope that it will be useful,          */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/* GNU General Public License for more details.                             */

/* You should have received a copy of the GNU General Public License        */
/* along with this program; if not, write to the Free Software              */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

/*--------------------------------------------------------------------------*/
/*  Name           : train_SVM.c                                            */
/*  Version        : 1.0                                                    */
/*  Creation       : 06/20/08                                               */
/*  Last update    : 10/25/25                                               */
/*  Subject        : Training algorithm of the LLW-M-SVM                    */
/*  Algo.          : Frank-Wolfe algorithm + decomposition method           */
/*  Author         : Yann Guermeur Yann.Guermeur@cnrs.fr                    */
/*--------------------------------------------------------------------------*/


#include <stdio.h>
#include <stdlib.h> 
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "algebre.h"

#define minimum(a,b) ((a)<=(b)?(a):(b))
#define maximum(a,b) ((a)>=(b)?(a):(b))

#define true 1
#define false 0
#define taille 81
#define very_small 1e-3
#define nb_iter 1e9
#define step 100

FILE *fs, *fc;

unsigned short xi[3];

int status;

long jump=false, i, j, k, l, m, dim_input, nb_data, iter, **X, *y, row, nb_symb,
y_i, y_j, Q=0, chunk_size=0, *table_chunk, *out_of_chunk, *in_chunk, active,
ind_pattern, nrand48(), random_num, rank, nb_SV=0;

char fichier_data[taille], fichier_alpha[taille], letter, commande[taille+20],
fichier_lp_sol[taille]="lp.res", fichier_dot_prod_aa[taille],
*ligne, *frag, fichier_fichcom[taille], fichier_alpha_0[taille],
*tampon[6] = {"00000","0000","000","00","0",""},
fichier_lp[taille]="lp.lp", fichier_vect_theta[taille];

double **alpha, C=0.0, QminusOneInv=0.0, **gradient, **delta, value,
obj_lp_lu=0.0, **lp_sol, theta_opt=0.0, *vect_theta, **dot_prod_aa,
partiel, **K, **H_delta, gradient_alpha, gradient_gamma, *rhs, gradient_diff;

/* Functions included in this program */

long indice_tampon(long parametre);
void caract_db();
void alloc_memory();
void read_data();
void read_alpha();
void check_feasible_sol();
void read_vect_theta();
void read_dot_prod_aa();
void compute_table_chunk();
void compute_K();
void compute_gradient();
void write_lp_file();
void read_lp_sol();
void check_opt_sol();
void compute_delta();
void compute_H_delta();
void compute_theta_opt();
void compute_new_alpha();
void write_vector(double **vector, char *fichier);

int main(int argc, char *argv[])

{

status = system("clear");
srand48(getpid());
strcpy(fichier_fichcom, argv[1]);

caract_db();
read_dot_prod_aa();
read_data();
read_alpha();
check_feasible_sol();
read_vect_theta();

for(iter=1; iter<=nb_iter; iter++)
  {

  if((iter%step) == 0)
    printf("\n\n*** Iteration: %6ld\n", iter);
  jump = false;
  compute_table_chunk();
  compute_K();
  compute_gradient();
  if(jump == false)
    {
    write_lp_file();
    status = system("./lp_solve < lp.lp > lp.res 2> lp.err");
    read_lp_sol();
    }
  if(jump == false)
    check_opt_sol();
  if(jump == false)
    {
    compute_delta();
    compute_theta_opt();
    compute_new_alpha();
    }
  if((iter%step) == 0)
    write_vector(alpha, fichier_alpha);

  } 

}

long indice_tampon(long parametre)

{

long valeur_long;

valeur_long = (long) floor(log10((double) parametre));
return valeur_long;

}

void caract_db()

{

if((fs=fopen(fichier_fichcom, "r"))==NULL)
  {
  printf("\nFile of parameters %s: cannot be open...\n", fichier_fichcom);
  exit(0);
  }

status = fscanf(fs, "%lf", &C);
status = fscanf(fs, "%ld", &chunk_size);

status = fscanf(fs, "%s", fichier_data);
printf("\nThe data file is: %s\n", fichier_data);
status = fscanf(fs, "%s", fichier_alpha_0);
status = fscanf(fs, "%s", fichier_alpha);
status = fscanf(fs, "%s", fichier_vect_theta);
status = fscanf(fs, "%s", fichier_dot_prod_aa);

fclose(fs);

sprintf(commande, "cp %s Save_alpha/.", fichier_alpha);

}

void alloc_memory()

{

X = matrix_l(nb_data, dim_input);
y = (long *) calloc(nb_data+1, sizeof(long));
alpha = matrix(nb_data, Q);
out_of_chunk = (long *) calloc(nb_data+1, sizeof(long));
in_chunk = (long *) calloc(nb_data+1, sizeof(long));
gradient = matrix(chunk_size, Q);
lp_sol = matrix(chunk_size, Q);
delta = matrix(chunk_size, Q);
table_chunk = (long *) calloc(chunk_size+1, sizeof(long));
K = matrix(chunk_size, nb_data);
H_delta = matrix(chunk_size, Q);
rhs = (double *) calloc(Q+1, sizeof(double));
vect_theta = (double *) calloc(dim_input+1, sizeof(double));
ligne = (char *) calloc(taille+1, sizeof(char));
frag = (char *) calloc(taille+1, sizeof(char));

}

void read_data()

{

long min_y, max_y;

if((fs=fopen(fichier_data, "r"))==NULL)
  {
  printf("\nData file %s: cannot be open...\n", fichier_data);
  exit(0);
  }

status = fscanf(fs, "%ld", &nb_data);
status = fscanf(fs, "%ld", &dim_input);
status = fscanf(fs, "%ld", &Q);

QminusOneInv = 1.0 / (Q - 1.0);

min_y = nb_data;
max_y = 0;

alloc_memory();

for(i=1; i<=nb_data; i++)
  {
  for(j=1; j<=dim_input; j++)
    {
    status = fscanf(fs, "%ld", &X[i][j]);
    if(X[i][j] == 0)
      X[i][j] = nb_symb;
    }
  status = fscanf(fs, "%ld", &y[i]);
  if(y[i] < min_y)
     min_y = y[i];
  if(y[i] > max_y)
    max_y = y[i];
  }

fclose(fs);

/*
printf("\nExtreme indices of categories : %2ld -> %2ld\n", min_y, max_y);
*/

if(((min_y != 0) && (min_y != 1)) || (max_y - min_y != Q-1))
  {
  printf("\nWrong numbering of the categories\n");
  exit(0);
  }

if(min_y == 0)
  {
  for(i=1; i<=nb_data; i++)
    y[i]++;
  }

}

void read_alpha()

{

if((fs=fopen(fichier_alpha_0, "r"))==NULL)
  {
  printf("\nFile of initial dual variable values %s: cannot be open...\n",
  fichier_alpha_0);
  exit(0);
  }

for(i=1; i<=nb_data; i++)
  for(k=1; k<=Q; k++)
    status = fscanf(fs, "%lf", &alpha[i][k]);

fclose(fs);

}

void check_feasible_sol()

{

double *constraints, norm;

for(i=1; i<=nb_data; i++)
  for(k=1; k<=Q; k++)
    if(((y[i] == k) && (alpha[i][k] != 0.0)) || (alpha[i][k] < 0.0)
       || (alpha[i][k] > C))
      {
      printf("\nNo feasible solution: alpha[%ld][%ld] = %lf\n\n",
             i, k, alpha[i][k]);
      exit(0);
      }

constraints = calloc(Q+1, sizeof(double));

for(k=1; k<=Q; k++)
  constraints[k] = 0.0;


for(i=1; i<=nb_data; i++)
  for(l=1; l<=Q; l++)
    for(k=1; k<=Q; k++)
      {
      constraints[k] += alpha[i][l] / (double)Q;
      if(k == l)
        constraints[k] -= alpha[i][l];
      }  

printf("\nSatisfaction of the equality constraints:\n\n");

for(k=1; k<=Q; k++)
  printf("%s%e\n", constraints[k] < 0.0 ? " " : "  ", constraints[k]);

norm = 0.0;

for(k=1; k<=Q; k++)
  norm += constraints[k] * constraints[k];

norm = sqrt(norm);

if(norm >= very_small)
  {
  printf("\nLarge deviation of the equality constraints...\n");
  exit(0);
  }

}

void read_vect_theta()

{

if((fs=fopen(fichier_vect_theta, "r"))==NULL)
  {
  printf("\nFile of the positional weighting %s: cannot be open...\n",
  fichier_vect_theta);
  exit(0);
  }
else
  printf("\nThe file of the positional weighting is: %s", fichier_vect_theta);

for(i=1; i<=dim_input; i++)
  status = fscanf(fs, "%lf", &vect_theta[i]);

fclose(fs);

}

void read_dot_prod_aa()

{

if((fs=fopen(fichier_dot_prod_aa, "r"))==NULL)
  {
  printf("\nFile of the substitution matrix %s: cannot be open...\n",
  fichier_dot_prod_aa);
  exit(0);
  }
else
  printf("The file of the substitution matrix is: %s\n", fichier_dot_prod_aa);

status = fscanf(fs, "%ld", &nb_symb);
dot_prod_aa = matrix(nb_symb, nb_symb);

for(i=1; i<=nb_symb; i++)
  for(j=1; j<=nb_symb; j++)
    status = fscanf(fs, "%lf", &dot_prod_aa[i][j]);

fclose(fs);

}

void write_lp_file()

{

/*
printf("\nEntering write_lp_file with a chunk of size %ld\n", chunk_size);
exit(0);
*/

if((fc=fopen(fichier_lp, "w"))==NULL)
  {
  printf("\nFile lp %s: cannot be open...\n", fichier_lp);
  exit(0);
  }

fprintf(fc, "min:");

for(i=1; i<=chunk_size; i++)
  {
  ind_pattern = table_chunk[i];
  y_i = y[ind_pattern];
  for(k=1; k<=Q; k++)
    if(k != y_i)
      {
      if(gradient[i][k] > 0.0)
        fprintf(fc, " +%lf A0%s%ld", gradient[i][k],
        tampon[indice_tampon(Q*(i-1)+k)], (Q*(i-1)+k));
      if(gradient[i][k] < 0.0)
        fprintf(fc, " %lf A0%s%ld", gradient[i][k],
        tampon[indice_tampon(Q*(i-1)+k)], (Q*(i-1)+k));
      }
  }

fprintf(fc, ";\n");

for(k=1; k<Q; k++)
  {
  rhs[k] = 0.0;
  for(i=1; i<=nb_data; i++)
    if(in_chunk[i] == 0)
      {
      for(l=1; l<=Q; l++)
        rhs[k] += alpha[i][l];
      rhs[k] -= Q * alpha[i][k];
      }
  }

for(k=1; k<Q; k++)
  {
  fprintf(fc, "CONST%1ld: ", k);

  for(i=1; i<=chunk_size; i++)
    {
    ind_pattern = table_chunk[i];
    y_i = y[ind_pattern];
    for(l=1; l<=Q; l++)
      if(l != y_i)
        {
        partiel = -1.0;
	if(l == k)
          partiel += Q;
        if(l != k)
          {
          fprintf(fc, "%f A0%s%ld ", partiel,
          tampon[indice_tampon(Q*(i-1)+l)],(Q*(i-1)+l));
          }
        else
          fprintf(fc, "+%f A0%s%ld ", partiel,
          tampon[indice_tampon(Q*(i-1)+l)], (Q*(i-1)+l));
	}
    }

  fprintf(fc, "= %lf;\n", rhs[k]);
  }

for(i=1; i<=chunk_size; i++)
  {
  ind_pattern = table_chunk[i];
  y_i = y[ind_pattern];
  for(k=1; k<=Q; k++)
    if(k != y_i)
      fprintf(fc, "A0%s%ld < %lf;\n",
      tampon[indice_tampon(Q*(i-1)+k)], (Q*(i-1)+k), C);
  }

fclose(fc);

}

void compute_table_chunk()

{

for(i=1; i<=nb_data; i++)
  {
  out_of_chunk[i] = i;
  in_chunk[i] = 0;
  }

if((iter%step) == 0)
  {
  nb_SV = 0;

  for(i=1; i<=nb_data; i++)
    {
    active = 0;
    k = 1;
    while((active == 0) && (k <= Q))
      {
      if((alpha[i][k] > 0.0) && (alpha[i][k] < C))
        {
        active = 1;
        nb_SV++;
        }
      k++;
      }
    }

  printf("\nNumber of margin Sup. Vec.: %ld", nb_SV);
  }

for(i=1; i<=chunk_size; i++)
  {
  if(i == 1)
    {
    row = iter % nb_data;
    if(row == 0)
      row = nb_data;
    }
  else
    {
    random_num = nrand48(xi);
    row = (random_num % (nb_data-i+1))+1;
    }
  table_chunk[i] = out_of_chunk[row];
  for(j=row; j<=nb_data-i; j++)
    out_of_chunk[j] = out_of_chunk[j+1];
  in_chunk[table_chunk[i]] = 1;
  }

}

void compute_K()

{

for(i=1; i<=chunk_size; i++)
  {
  ind_pattern = table_chunk[i];
  for(j=1; j<=nb_data; j++)
  K[i][j] = gaussian(X[ind_pattern], X[j], vect_theta, dot_prod_aa, dim_input);
  }

}

void compute_gradient()

{

double partial_sum_alpha = 0.0;

for(i=1; i<=chunk_size; i++)
  for(k=1; k<=Q; k++)
    {
    if(k != y[table_chunk[i]])
      {
      gradient[i][k] = - QminusOneInv;
      for(j=1; j<=nb_data; j++)
        {
	partial_sum_alpha = 0.0;

        for(l=1; l<=Q; l++)
          partial_sum_alpha += alpha[j][l];

	partiel = alpha[j][k] - (partial_sum_alpha / Q);
	gradient[i][k] += partiel * K[i][j];
        }
      }
    }

}

void compute_H_delta()

{

double partial_sum_delta = 0.0;

/* printf("\nMatrix H delta...\n"); */

for(i=1; i<=chunk_size; i++)
  for(k=1; k<=Q; k++)
    {
    H_delta[i][k] = 0.0; 
    if(k != y[table_chunk[i]])
      {
      for(j=1; j<=chunk_size; j++)
        {
        partial_sum_delta = 0.0;

        for(l=1; l<=Q; l++)
          partial_sum_delta += delta[j][l];

        partiel = delta[j][k] - (partial_sum_delta / Q);
	H_delta[i][k] += partiel * K[i][table_chunk[j]];
	}
      }
/*
    printf("%s%e %c",  H_delta[i][k] < 0.0 ? " " : "  ", H_delta[i][k], 
		    (k == Q) ? '\n': ' ');
*/
    }

}

void read_lp_sol()

{

if((fs=fopen(fichier_lp_sol, "r"))==NULL)
  {
  printf("\nThe file %s of the LP solution cannot be open...\n",
  fichier_lp_sol);
  exit(0);
  }

if(fgets(ligne, taille, fs)==NULL)
  exit(0);
else
  if(strlen(ligne) > 1)
    {
/*
    printf("%s",ligne);
    Pause("");
*/
    jump=true;
    }

if(jump == false)
  {
  if(fgets(ligne, taille, fs)==NULL)
    exit(0);

  if(strncmp("Value of objective function:", ligne, 28) != 0)
    {
/*
    printf("%s",ligne);
    Pause("");
*/
    jump=true;
    }
  else
    {
    frag = strtok(ligne, ":");
    frag = strtok(NULL, ":");
    obj_lp_lu = atof(frag);
/*
    printf("\nValue of objective function: %lf", obj_lp_lu);
    Pause("");
*/
    if(fgets(ligne, taille, fs)==NULL)
      exit(0);
    if(fgets(ligne, taille, fs)==NULL)
      exit(0);

    for(i=1; i<=chunk_size; i++)
      {
      ind_pattern = table_chunk[i];
      y_i = y[ind_pattern];
      lp_sol[i][y_i]=0.0;
      }

    for(i=1; i<=chunk_size; i++)
      for(k=1; k<Q; k++)
        {
        status = fscanf(fs, "%c", &letter);
        status = fscanf(fs, "%ld", &rank);
        status = fscanf(fs, "%lf", &value);
        j = (rank-1)/Q+1;
        l = (rank-1)%Q+1;
        lp_sol[j][l] = value;
        status = fscanf(fs, "%c", &letter);
        }
    }
  }

fclose(fs);

for(i=1; i<=chunk_size; i++)
  for(k=1; k<=Q; k++)
    if((lp_sol[i][k] < 0.0) || (lp_sol[i][k] > C))
      jump = true;

/*
if(jump == false)
  {
  printf("\nFeasible solution of the LP problem...\n");
  }
*/

}

void check_opt_sol()

{

gradient_alpha = 0.0; 
gradient_gamma = 0.0;
gradient_diff = 0.0;

for(i=1; i<=chunk_size; i++)
  {
  ind_pattern = table_chunk[i];
  y_i = y[ind_pattern];

  for(k=1; k<=Q; k++)
    if(k != y_i)
      {
      gradient_alpha += gradient[i][k] * alpha[ind_pattern][k];
      gradient_gamma += gradient[i][k] * lp_sol[i][k];
      }
  }

gradient_diff = gradient_alpha - gradient_gamma;

if(gradient_diff <= 0.0)
  jump = true;

/*
if(jump == false)
  {
  printf("\nOptimal solution of the LP problem: gradient_diff = %e...\n",
	 gradient_diff);
  }
*/

}

void compute_new_alpha()

{

for(i=1; i<=chunk_size; i++)
  {
  ind_pattern = table_chunk[i];
  for(k=1; k<=Q; k++)
    {
    alpha[ind_pattern][k] *= (1.0-theta_opt);
    alpha[ind_pattern][k] += theta_opt * lp_sol[i][k];
    }
  }

}

void compute_delta()

{

/* printf("\n\nValues of parameters delta...\n"); */

for(i=1; i<=chunk_size; i++)
  {
  ind_pattern = table_chunk[i];
  y_i = y[ind_pattern];
  delta[i][y_i] = 0.0;

  for(k=1; k<=Q; k++)
    {
    if(k != y_i)
      delta[i][k] = alpha[ind_pattern][k] - lp_sol[i][k];
/*
    printf("%s%e %c", delta[i][k] < 0.0 ? " " : "  ", delta[i][k], 
    (k == Q) ? '\n': ' ');
*/
    }
  }

/* exit(0); */

}

void compute_theta_opt()

{

double denominateur = 0.0;

compute_H_delta();

for(i=1; i<=chunk_size; i++)
  for(k=1; k<=Q; k++)
    denominateur += delta[i][k] * H_delta[i][k];

if(denominateur <= 0.0)
  {
  printf("\nWrong estimate of the quadratic form: %e\n", denominateur);
  exit(0);
  }

theta_opt = gradient_diff / denominateur;

if(theta_opt > 1.0)
  theta_opt = 1.0;

/*
if((iter%step) == 0)
  printf("\n    Optimal value of theta: %10.8f\n", theta_opt);
*/

}

void write_vector(double **vector, char *fichier)

{

if((fc=fopen(fichier, "w"))==NULL)
  {
  printf("\nThe file %s cannot be open...\n", fichier);
  exit(0);
  }

for(i=1; i<=nb_data; i++)
  for(k=1; k<=Q; k++)
    fprintf(fc, "%e %c", vector[i][k], (k==Q) ? '\n': ' ');

fclose(fc);

status = system(commande);

}
