/* Copyright 2000-2023 Yann Guermeur                                        */

/* This program is free software; you can redistribute it and/or modify     */
/* it under the terms of the GNU General Public License as published by     */
/* the Free Software Foundation; either version 2 of the License, or        */
/* (at your option) any later version.                                      */

/* This program is distributed in the hope that it will be useful,          */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/* GNU General Public License for more details.                             */

/* You should have received a copy of the GNU General Public License        */
/* along with this program; if not, write to the Free Software              */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

/*--------------------------------------------------------------------------*/
/*  Name           : eval_SVM.c                                             */
/*  Version        : 1.1                                                    */
/*  Creation       : 04/30/00                                               */
/*  Last update    : 08/08/25                                               */
/*  Subject        : Implementation of the M-SVM of Weston and Watkins      */
/*  Module         : M-SVM used in test                                     */
/*  Author         : Yann Guermeur (Yann.Guermeur@cnrs.fr)                  */
/*--------------------------------------------------------------------------*/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "algebre.h"
#include "biblio.h"

#define true 1
#define false 0
#define min(a,b) ((a)<=(b)?(a):(b))
#define max(a,b) ((a)<=(b)?(b):(a))

#define taille 81
#define very_small 3e-4
#define step 100

FILE *fs, *fc;

int status;

long **X_app, **X_test, *y_app, *y_test, smallest_cat=0, largest_cat=0, nb_symb,
Q=0, i, j, k, l, dim_input, nb_data_app=0, nb_data_test=0, y_i, y_j, 
minimal_entry, *in_path, *sequence_of_categories, index_of_transition,
direct_path=0, **mat_conf, **margin_vect;

char fichier_app[taille], fichier_b[taille],
fichier_alpha[taille], fichier_dot_prod_aa[taille], fichier_SV[taille],
fichier_test[taille], commande[taille+20], fichier_vect_theta[taille],
fichier_fichcom[taille], fichier_resultat[taille], var_open = '(',
var_close = ')';

double R_emp=0.0, **alpha, **delta_b_SVM, *b_SVM, C=0.0, partiel,
primal=0.0, dual=0.0, Q_form_dual=0.0, **gradient, max_alpha=0.0,
cost=0.0, **H_alpha, *vect_theta, **dot_prod_aa;

/* Functions included in this program */

void caract_db();
void read_data();
void alloc_memory();
void read_alpha();
void check_feasible_sol();
void check_margin_vect();
void read_vect_theta();
void read_dot_prod_aa();
void compute_gradient();
void explore();
void depth_first_search();
void estimate_b();
void standardize_b();
void write_b();
void eval_training();
double fonction(long categorie, long *vecteur);
void compute_outputs(long **X, long *y, long nb_data);
void compute_obj_dual();
void compute_obj_primal();

int main(int argc, char *argv[])

{

strcpy(fichier_fichcom, argv[1]);

status = system("clear");
caract_db();
read_dot_prod_aa();
read_data();
alloc_memory();
read_alpha();
read_vect_theta();

if(max_alpha > 0.0)
  {
  check_feasible_sol();
  check_margin_vect();
  depth_first_search();
  compute_gradient();
  estimate_b();
  write_b();
  }

eval_training();
compute_obj_dual();
compute_obj_primal();

printf("\n*** Training performance\n");
printf("\n            Dual objective function: %e", dual);
printf("\nEstimated primal objective function: %e", primal);

if(primal != 0.0)
  printf("\n%cratio %5.3lf%c\n", var_open, dual / primal, var_close);

display_stats(fichier_app, nb_data_app, Q, mat_conf, cost);

Pause("\a=> New Line to access the test performance...");

printf("\n*** Test performance\n");
compute_outputs(X_test, y_test, nb_data_test);

display_stats(fichier_test, nb_data_test, Q, mat_conf, cost);

}

void standardize_b()

{

double sum_b = 0.0;

for(k=1; k<=Q; k++)
  sum_b += b_SVM[k];

sum_b /= Q;

for(k=1; k<=Q; k++)
  b_SVM[k] -= sum_b;

}

void write_b()

{

if((fc=fopen(fichier_b, "w"))==NULL)
  {
  printf("\nFile of vector b: %s cannot be open...\n", fichier_b);
  exit(0);
  }

for(k=1; k<=Q; k++)
  fprintf(fc, "%12.6lf\n", b_SVM[k]); 

fclose(fc);

}

void caract_db()

{

if((fs=fopen(fichier_fichcom, "r"))==NULL)
  {
  printf("\nFile of parameters: %s cannot be open...\n", fichier_fichcom);
  exit(0);
  }

status = fscanf(fs, "%lf", &C);
status = fscanf(fs, "%s", fichier_app);
printf("\nThe file of the training set is: %s", fichier_app);
status = fscanf(fs, "%s", fichier_test);
printf("\n    The file of the test set is: %s\n", fichier_test);

status = fscanf(fs, "%s", fichier_alpha);
status = fscanf(fs, "%s", fichier_vect_theta);
status = fscanf(fs, "%s", fichier_dot_prod_aa);
status = fscanf(fs, "%s", fichier_SV);
status = fscanf(fs, "%s", fichier_resultat);
status = fscanf(fs, "%s", fichier_b);

fclose(fs);

}

void read_data()

{

long min_y, ind_min_y, max_y, ind_max_y, *cardinal_cat, cardinal_min, 
cardinal_max=0;

if((fs=fopen(fichier_app, "r"))==NULL)
  {
  printf("\nFile of data: %s cannot be open...\n", fichier_app);
  exit(0);
  }

status = fscanf(fs, "%ld", &nb_data_app);
status = fscanf(fs, "%ld", &dim_input);
status = fscanf(fs, "%ld", &Q);

mat_conf = matrix_l(Q,Q);
cardinal_cat = (long *) calloc(Q+1, sizeof(long));

X_app = matrix_l(nb_data_app, dim_input);
y_app = (long *) calloc(nb_data_app+1, sizeof(long));

min_y = nb_data_app;
max_y = 0;

for(i=1; i<=nb_data_app; i++)
  {
  for(j=1; j<=dim_input; j++)
    {
    status = fscanf(fs, "%ld", &X_app[i][j]);
    if(X_app[i][j] == 0)
      X_app[i][j] = nb_symb;
    }
  status = fscanf(fs, "%ld", &y_app[i]);
  if(y_app[i] < min_y)
    {
    min_y = y_app[i];
    ind_min_y = i;
    }
  if(y_app[i] > max_y)
    {
    max_y = y_app[i];
    ind_max_y = i;
    }
  }

fclose(fs);

if((fs=fopen(fichier_test, "r"))==NULL)
  {
  printf("\nFile of data: %s cannot be open...\n", fichier_test);
  exit(0);
  }

status = fscanf(fs, "%ld", &nb_data_test);
status = fscanf(fs, "%ld", &dim_input);
status = fscanf(fs, "%ld", &Q);

X_test = matrix_l(nb_data_test, dim_input);
y_test = (long *) calloc(nb_data_test+1, sizeof(long));

for(i=1; i<=nb_data_test; i++)
  {
  for(j=1; j<=dim_input; j++)
    {
    status = fscanf(fs, "%ld", &X_test[i][j]);
    if(X_test[i][j] == 0)
      X_test[i][j] = nb_symb;
    }
  status = fscanf(fs, "%ld", &y_test[i]);
  if(y_test[i] < min_y)
    min_y = y_test[i];
  if(y_test[i] > max_y)
    max_y = y_test[i];
  }

fclose(fs);

/*
printf("\nExtreme indices of categories : %2ld -> %2ld\n", min_y, max_y);
*/

if(((min_y != 0) && (min_y != 1)) || (max_y - min_y != Q-1))
  {
  printf("\nWrong numbering of the categories\n");
  exit(0);
  }

if(min_y == 0)
  {
  for(i=1; i<=nb_data_app; i++)
    y_app[i]++;
  for(i=1; i<=nb_data_test; i++)
    y_test[i]++;
  }

for(k=1; k<=Q; k++)
  cardinal_cat[k] = 0;

for(i=1; i<=nb_data_app; i++)
  cardinal_cat[y_app[i]]++;

cardinal_min = nb_data_app;

for(k=1; k<=Q; k++)
  {
  if(cardinal_cat[k] > cardinal_max)
    {
    cardinal_max = cardinal_cat[k];
    largest_cat = k;
    }
  if(cardinal_cat[k] < cardinal_min)
    {
    cardinal_min = cardinal_cat[k];
    smallest_cat = k;
    }
  }
  
/*
for(k=1; k<=Q; k++)
  printf("\nCardinal of category %2ld: %5ld", k, cardinal_cat[k]);

printf("\nIndex of the smallest category: %2ld", smallest_cat);
printf("\nIndex of the largest category: %2ld", largest_cat);
Pause("");
*/

if(cardinal_min == 0)
  {
  printf("\nNo training example for category %2ld\n", smallest_cat);
  exit(0);
  }

}

void alloc_memory()

{

alpha = matrix(nb_data_app, Q);
vect_theta = (double *) calloc(dim_input+1, sizeof(double));
margin_vect = matrix_l(Q, Q);

for(k=1; k<=Q; k++)
  for(l=1; l<=Q; l++)
    margin_vect[k][l] = 0;

delta_b_SVM = matrix(Q, Q);
b_SVM = (double *) calloc(Q+1, sizeof(double));

for(k=1; k<=Q; k++)
  b_SVM[k] = 0.0;

for(k=1; k<=Q; k++)
  for(l=1; l<=Q; l++)
    delta_b_SVM[k][l] = 0.0;

gradient = matrix(nb_data_app, Q);
H_alpha  = matrix(nb_data_app, Q);

for(i=1; i<=nb_data_app; i++)
    for(k=1; k<=Q; k++)
      H_alpha[i][k] = 0.0;

for(i=1; i<=nb_data_app; i++)
  {
  y_i = y_app[i];
  for(k=1; k<=Q; k++)
    if(k != y_i)
      gradient[i][k] = -1.0;
    else
      gradient[i][k] = 0.0;
  }

}

void read_alpha()

{

max_alpha = 0.0;

if((fs=fopen(fichier_alpha, "r"))==NULL)
  {
  printf("\nFile of dual variables: %s cannot be open...\n", 
  fichier_alpha);
  exit(0);
  }

alpha = matrix(nb_data_app, Q);

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
    {
    status = fscanf(fs, "%lf", &alpha[i][k]);
    if((alpha[i][k] != 0.0) && (k == y_app[i]))
      {
      printf("\nEx. %ld: pb. with the dummy variable...\n\n", i);
      exit(0);
      }
    if(alpha[i][k] > max_alpha)
      max_alpha = alpha[i][k];
    }

fclose(fs);

if(max_alpha < C)
  printf("\nMaximum value of a dual variable, %e, inferior to C (= %e) \n", max_alpha, C);

}

void check_feasible_sol()

{

double *constraints, norm;

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
    if((alpha[i][k] < 0.0) || (alpha[i][k] > C))
       {
       printf("\nNo feasible solution: alpha[%ld][%ld] = %lf\n\n",
              i, k, alpha[i][k]);
       exit(0);
       }
     
constraints = calloc(Q+1, sizeof(double));

for(k=1; k<=Q; k++)
  constraints[k] = 0.0;

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
     {
     if(y_app[i] == k)
       for(l=1; l<=Q; l++)
         constraints[k] -= alpha[i][l];
     else
       constraints[k] += alpha[i][k];  
     }

printf("\n\nSatisfaction of the equality constraints:\n\n");

for(k=1; k<=Q; k++)
  printf("%s%e\n", constraints[k] < 0.0 ? " " : "  ", constraints[k]);

norm = 0.0;

for(k=1; k<=Q; k++)
  norm += constraints[k] * constraints[k];

norm = sqrt(norm);

if(norm >= very_small)
  {
  printf("\nLarge deviation of the equality constraints...\n");
  exit(0);
  }
else
  {
  sprintf(commande, "cp %s Feasible/.", fichier_alpha);
  status = system(commande);
  }

}

void read_vect_theta()

{

if((fs=fopen(fichier_vect_theta, "r"))==NULL)
  {
  printf("\nFile of the positional weighting %s: cannot be open...\n",
  fichier_vect_theta);
  exit(0);
  }
else
  printf("\nThe file of the positional weighting is: %s", fichier_vect_theta);

for(i=1; i<=dim_input; i++)
  status = fscanf(fs, "%lf", &vect_theta[i]);

fclose(fs);

}

void read_dot_prod_aa()

{

if((fs=fopen(fichier_dot_prod_aa, "r"))==NULL)
  {
  printf("\nFile of the substitution matrix %s: cannot be open...\n",
  fichier_dot_prod_aa);
  exit(0);
  }
else
  printf("\n The file of the substitution matrix is: %s", fichier_dot_prod_aa);

status = fscanf(fs, "%ld", &nb_symb);
dot_prod_aa = matrix(nb_symb, nb_symb);

for(i=1; i<=nb_symb; i++)
  for(j=1; j<=nb_symb; j++)
    status = fscanf(fs, "%lf", &dot_prod_aa[i][j]);

fclose(fs);

}

void compute_gradient()

{

printf("\nStart of the gradient computation...\n");

for(i=1; i<=nb_data_app; i++)
  {
  if((i%step) == 0)
    printf("\n%9ld", i);
  y_i = y_app[i];

  for(k=1; k<=Q; k++)
    {
    H_alpha[i][k] = 0.0;
    if(k != y_i)
      {
      for(j=1; j<=nb_data_app; j++)
        {
        y_j = y_app[j];
        partiel = 0.0;

        if(y_j == y_i)
          for(l=1; l<=Q; l++)
            partiel += alpha[j][l];

        if(y_j == k)
          for(l=1; l<=Q; l++)
            partiel -= alpha[j][l];

        partiel += alpha[j][k] - alpha[j][y_i];
          
        if(partiel != 0.0)
          H_alpha[i][k] += partiel * 
          gaussian(X_app[i], X_app[j], vect_theta, dot_prod_aa, dim_input);
        }
      }
    }
  }

for(i=1; i<=nb_data_app; i++)
  {
  y_i = y_app[i];
  for(k=1; k<=Q; k++)
    if(k != y_i)
      gradient[i][k] = H_alpha[i][k] - 1.0;
    else
      gradient[i][k] = 0.0;
  }

printf("\n\nEnd of the gradient computation...\n");

}

void check_margin_vect()

{

long ind_min, ind_max;
double center = C/2.0, **dist_min, current_dist;

dist_min = matrix(Q,Q);

for(k=1; k<=Q; k++)
  for(l=1; l<=Q; l++)
    {
    dist_min[k][l] = center;
    margin_vect[k][l] = 0;
    }

for(i=1; i<=nb_data_app; i++)
  {
  y_i = y_app[i];
  for(k=1; k<=Q; k++)
    {
    current_dist = fabs(alpha[i][k] - center);
    ind_min = min(y_i,k);
    ind_max = max(y_i,k);

    if(current_dist < dist_min[ind_min][ind_max])
      {
      margin_vect[ind_min][ind_max] = i;
      margin_vect[ind_max][ind_min] = i;
      dist_min[ind_min][ind_max] = current_dist;
      }
    }
  }

minimal_entry = nb_data_app;

for(k=1; k<Q; k++)
  for(l=k+1; l<=Q; l++)
    if(margin_vect[k][l] == 0)
      {
      minimal_entry = 0;
/*
      printf("\nMissing margin SV for the pair of categories (%2ld, %2ld).\n",
      k, l);
*/
      }

if(minimal_entry == 0)
  {
  printf("\nBoundaries without margin SV\n");

  display_mat(margin_vect, Q, Q, (long) ceil(log10((double) nb_data_app)));
  Pause("\a=> New Line to access the training performance...");
  }

}

double fonction(long categorie, long *vecteur)

{

long indice1, indice2;

double resultat = 0.0;

for(indice1=1; indice1<=nb_data_app; indice1++)
  {
  if(y_app[indice1] == categorie)
    {
    partiel = 0.0;
    for(indice2=1; indice2<=Q; indice2++)
      partiel += alpha[indice1][indice2];
    }
  else
    partiel = - alpha[indice1][categorie];

  if(partiel != 0.0)
    resultat += partiel * gaussian(X_app[indice1], vecteur, vect_theta, dot_prod_aa, dim_input);
  }

return resultat;

}

void eval_training()

{

long selected_cat = 0;

double delta, discriminant_value;

for(k=1; k<=Q; k++)
  for(l=1; l<=Q; l++)
    mat_conf[k][l] = 0;

R_emp = 0.0;

for(i=1; i<=nb_data_app; i++)
  {
  discriminant_value = -1.0;
  y_i = y_app[i];
  selected_cat = y_i;

  for(k=1; k<=Q; k++)
    if(k != y_i)
      {
      delta = gradient[i][k] + b_SVM[y_i] - b_SVM[k];
      if(delta < 0.0)
         R_emp -= delta;
      if(delta <= discriminant_value)
        {
        discriminant_value = delta;
        selected_cat = k;
        }
      }

  if(selected_cat == y_i)
    mat_conf[y_i][y_i]++;
  else
    if(discriminant_value == -1.0)
      mat_conf[y_i][largest_cat]++;
    else
      mat_conf[y_i][selected_cat]++;
  }
  
}

void explore(long vertex)

{

long vertex2;

for(vertex2=1; vertex2<=Q; vertex2++)
  if(margin_vect[vertex][vertex2] != 0)
    {
    if(in_path[vertex2] == 0)
      {
      index_of_transition++;
      in_path[vertex] = 1;
      in_path[vertex2] =1;
      sequence_of_categories[2*index_of_transition-1] = vertex;
      sequence_of_categories[2*index_of_transition] = vertex2;
      explore(vertex2);      
      }
    }
  
}

void depth_first_search()

{

/* printf("\nComputing a path connecting the categories...\n"); */
in_path = (long *) calloc(Q+1, sizeof(long));
sequence_of_categories = (long *) calloc(2*(Q+1), sizeof(long));

index_of_transition = 0;

for(k=1; k<=Q; k++)
  in_path[k] = 0;

for(k=1; k<=Q; k++)
  if(in_path[k] == 0)
    explore(k);

/*
for(k=1; k<=Q-1;k++)
  if(sequence_of_categories[2*k-1] != 0)
    printf("\nfrom %2ld to %2ld",
    sequence_of_categories[2*k-1], sequence_of_categories[2*k]);
*/

for(k=1; k<=Q; k++)
  if(in_path[k] == 0)
    {
    printf("\n\nThe graph of categories is not connected.\n\n");
    exit(0);
    }

direct_path = 1;

for(k=2; k<=Q; k++)
  if(margin_vect[1][k] == 0)
    direct_path = 0;

if(direct_path == 1)
  printf("\nRobust estimation of the biases...\n");
else
  printf("\nAdvanced estimation of the biases...\n");

}

void estimate_b()

{

long ind_data, ind_min, ind_max, cat_left, cat_right;

b_SVM[sequence_of_categories[1]] = 0.0;

if((fc=fopen(fichier_SV, "w"))==NULL)
  {
 printf("\nFile of margin support vectors: %s cannot be open...\n", fichier_SV);
  exit(0);
  }

if(direct_path == 1)
  {
  for(l=2; l<=Q; l++)
    {
    ind_data = margin_vect[1][l];
    fprintf(fc, "Example %5ld, SV between categories 1 and %2ld\n",
      ind_data, l);

    if(y_app[ind_data] == 1)
      b_SVM[l] = gradient[ind_data][l];
    else
      b_SVM[l] = -gradient[ind_data][1];
    }
  }
else
  {
  printf("\nPath followed to derive the values of the biases:");
  for(k=1; k<Q; k++)
    {
    cat_left = sequence_of_categories[2*k-1];
    cat_right = sequence_of_categories[2*k];
    ind_min = min(cat_left, cat_right);
    ind_max = max(cat_left, cat_right);
    ind_data = margin_vect[ind_min][ind_max];
    fprintf(fc, "Example %5ld, SV between categories %2ld and %2ld\n",
      ind_data, cat_left, cat_right);

    if(y_app[ind_data] == cat_right)
      b_SVM[cat_right] = -gradient[ind_data][cat_left];
    else
      b_SVM[cat_right] = gradient[ind_data][cat_right];
    b_SVM[cat_right] += b_SVM[cat_left];
    }
  }

fclose(fc);

standardize_b();

}

void compute_outputs(long **X, long *y, long nb_data)

{

long ind_max = 0;

double *output, maximum;

output = (double *) calloc(Q+1, sizeof(double));

for(k=1; k<=Q; k++)
  for(l=1; l<=Q; l++)
    mat_conf[k][l] = 0;

if((fc=fopen(fichier_resultat, "w"))==NULL)
  {
  printf("\nFile of outputs: %s cannot be open...\n", fichier_resultat);
  exit(0);
  }

for(i=1; i<=nb_data; i++)
  {
  for(k=1; k<=Q; k++)
    output[k] = fonction(k, X[i]) + b_SVM[k];

  ind_max = 1;
  maximum = output[1];

  for(k=2; k<=Q; k++)
    if(output[k] > maximum)
      {
      maximum = output[k];
      ind_max = k;
      }

  if(output[largest_cat] == maximum)
    ind_max = largest_cat;

  mat_conf[(int)y[i]][(int)ind_max]++;

  for(k=1; k<=Q; k++) 
    fprintf(fc, "%12.6f%c", output[k], (k==Q) ? '\n' : ' ');

  if((i % step) == 0)
    printf("\nExample: %5ld", i);
  }

fclose(fc);

printf("\n");

}

void compute_obj_dual()

{

Q_form_dual = 0.0;

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
    Q_form_dual += H_alpha[i][k] * alpha[i][k];

Q_form_dual *= -0.5;

dual = Q_form_dual;

for(i=1; i<=nb_data_app; i++)
  for(k=1; k<=Q; k++)
    dual += alpha[i][k];

}

void compute_obj_primal()

{

primal = - Q_form_dual + C * R_emp;

if(primal < dual)
  {
  printf("\nInconsistency in the values of the objective functions...\n\n");
  exit(0);
  }

}
