
/*----------------------------------------------------------------------------*/
/*  Name           : tune_kernel.c                                            */
/*  Version        : 1.0                                                      */
/*  Creation       : 04/16/02                                                 */
/*  Last update    : 09/26/25                                                 */
/*  Subject        : Tuning kernel parameters with kernel target alignment    */
/*  Algo.          : Gradient descent method + decomposition method           */
/*  Author         : Regis Vert and Yann Guermeur                             */
/*----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include "algebre.h"
#include <time.h>

#define taille 81
#define nb_iter 100000
#define step 1000

FILE *fs, *fc;

unsigned short xi[3];

int status;

long  chunk_size, dim_input, ind_theta, iter, nb_symb, nb_data, Q,
i, j, k, l, *out_of_chunk, **X, *y, *chunk, nrand48(), random_num, rank;

char fichier_data[taille], fichier_theta[taille], fichier_theta_0[taille], 
fichier_D[taille], fichier_fichcom[taille];

double *theta, *theta_2, *grad_J_theta, mu=1.0, **dot_prod_aa,
value, initial_value=0.0, final_value=0.0,
**K_t, **K_theta, **grad_K_theta, K_theta_dot_K_t=0.0, K_t_dot_K_t=0.0,
K_theta_dot_K_theta=0.0, grad_K_theta_dot_K_t=0.0, ratio=0.0, **D_2,
grad_K_theta_dot_K_theta=0.0, epsilon, norm_theta=0.0, norm_grad_J=0.0;

/* procedures called in this programme */

void alloc_memory();
void caract_db();
void read_data();
void read_theta();
void read_D();
void compute_chunk();
void compute_K_t();
void compute_K_theta();
void compute_grad_K_theta(long position);
void compute_grad_J();
double compute_objective_function();
void compute_epsilon();
void update_theta();
void write_theta();

int main(int argc, char *argv[])

{

status = system("clear");
strcpy(fichier_fichcom, argv[1]);
caract_db();
read_data();
read_theta();
read_D();

for(iter=1; iter<=nb_iter; iter++)
  {

  if((iter%step)==0)
    {
    printf("\n*************************");
    printf("\nIteration %6ld ", iter);
    printf("\n*************************\n");
    }

  compute_chunk();
  compute_grad_J();
  compute_epsilon();
  update_theta();

  if((iter % step)==0)
    write_theta();

  }

}

void caract_db()

{

if((fs=fopen(fichier_fichcom, "r"))==NULL)
  {
  printf("\nFile of parameters %s: cannot be open...\n", fichier_fichcom);
  exit(0);
  }

status = fscanf(fs, "%ld", &nb_symb);
status = fscanf(fs, "%ld", &Q);
status = fscanf(fs, "%ld", &chunk_size);
/* status = fscanf(fs, "%lf", &mu); */
status = fscanf(fs, "%s", fichier_data);
status = fscanf(fs, "%s", fichier_theta_0);
status = fscanf(fs, "%s", fichier_theta);
status = fscanf(fs, "%s", fichier_D);

fclose(fs);

/* Pause("Parameters have been read..."); */

}

void alloc_memory()

{

dot_prod_aa = matrix(nb_symb, nb_symb);
D_2 = matrix(nb_symb, nb_symb);
theta = (double *) calloc(dim_input+1, sizeof(double));
theta_2 = (double *) calloc(dim_input+1, sizeof(double));
grad_J_theta = (double *) calloc(dim_input+1, sizeof(double));
X = matrix_l(nb_data, dim_input);
y = (long *) calloc(nb_data+1, sizeof(long));
chunk = (long *) calloc(chunk_size+1, sizeof(long));
K_t = matrix(chunk_size, chunk_size);
K_theta = matrix(chunk_size, chunk_size);
grad_K_theta = matrix(chunk_size, chunk_size);
out_of_chunk = (long *) calloc(nb_data+1, sizeof(long));

/* Pause("Memory has been allocated"); */

}


void read_data()

{

long y_min = Q, y_max = 0;

if((fs=fopen(fichier_data, "r"))==NULL)
  {
  printf("\nData file %s: cannot be open...\n", fichier_data);
  exit(0);
  }

status = fscanf(fs, "%ld", &nb_data);
status = fscanf(fs, "%ld", &dim_input);
status = fscanf(fs, "%ld", &Q);

alloc_memory();

for(i=1; i<=nb_data; i++)
  {
  for(j=1; j<=dim_input; j++)
    {
    status = fscanf(fs, "%ld", &X[i][j]);
    if(X[i][j] == 0)
      X[i][j] = nb_symb;
    }
  status = fscanf(fs, "%ld", &y[i]);
 
  if(y[i] < y_min)
    y_min = y[i];

  if(y[i] > y_max)
    y_max = y[i];
  }

fclose(fs);

if((y_min != 0) && (y_min != 1))
  {
  printf("\nWrong numbering of the categories: y_min = %ld,\n", y_min);
  exit(0);
  }

if(y_max - y_min != Q-1)
  {
  printf("\nWrong numbering of the categories: y_max - y_min = %ld,\n",
	 y_max - y_min);
  exit(0);
  }

if(y_min == 0)
  for(i=1; i<=nb_data; i++)
    y[i]++;

/*
printf("\nData has been read...\n");
*/

}

void read_theta()

{

if((fs=fopen(fichier_theta_0, "r"))==NULL)
  {
  printf("\nFile of initial vector theta %s: cannot be open...\n", fichier_theta_0);
  exit(0);
  }

for(i=1; i<=dim_input; i++)
  {
  status = fscanf(fs, "%lf", &value);
  theta_2[i] = value / mu;
  theta[i] = sqrt(theta_2[i]);
  }

fclose(fs);

/*
printf("\n*** Initial vector theta\n");

for(i=1; i<=dim_input; i++)
  printf("%lf\n", theta[i]);
*/

}

void read_D()

{

if((fs=fopen(fichier_D, "r"))==NULL)
  {
  printf("\nFile of distances %s: cannot be open...\n", fichier_D);
  exit(0);
  }

status = fscanf(fs, "%ld", &nb_symb);

for(k=1; k<=nb_symb; k++)
  for(l=1; l<=nb_symb; l++)
    status = fscanf(fs, "%lf", &dot_prod_aa[k][l]);

fclose(fs);

for(k=1; k<=nb_symb; k++)
  for(l=1; l<=nb_symb; l++)
    D_2[k][l] = dot_prod_aa[k][k] + dot_prod_aa[l][l] - 2.0 * dot_prod_aa[k][l];

/*
printf("\nMatrix of dot products between amino acids read...\n\n");
*/

}

void compute_chunk()

{

for(i=1; i<=nb_data; i++)
  out_of_chunk[i] = i;

for(i=1; i<=chunk_size; i++)
  {
  if(i==1)
    {
    rank = iter % nb_data;
    if(rank == 0);
      rank = nb_data;
    }
  else
    {
    random_num = nrand48(xi);
    rank = (random_num % (nb_data-i+1))+1;
    }
  chunk[i] = out_of_chunk[rank];

  for(j=rank; j<=nb_data-i; j++)
    out_of_chunk[j] = out_of_chunk[j+1];
  }

}

void compute_K_t()

{

for(i=1; i<=chunk_size; i++)
  {
  K_t[i][i] = 1.0;
  for(j=i+1; j<=chunk_size; j++)
    {
    K_t[i][j] = (y[chunk[i]] == y[chunk[j]]) ? 1.0 : 0.0;
    K_t[j][i] = K_t[i][j];
    }
  }

}

void compute_K_theta()

{

for(i=1; i<=chunk_size; i++)
  {
  K_theta[i][i] = 1.0;
  for(j=i+1; j<=chunk_size; j++)
     {
     K_theta[i][j] =
     gaussian(X[chunk[i]], X[chunk[j]], mu, theta_2, dot_prod_aa, dim_input);
     K_theta[j][i] = K_theta[i][j];
     }
  }

}

void compute_grad_K_theta(long position)

{

for(i=1; i<=chunk_size; i++)
  {
  grad_K_theta[i][i] = 0.0;
  for(j=i+1; j<=chunk_size; j++)
    {
    grad_K_theta[i][j] = -2.0 * mu * theta[position] *
    D_2[X[chunk[i]][position]][X[chunk[j]][position]] *
    K_theta[i][j];
    grad_K_theta[j][i] = grad_K_theta[i][j];
    }
  }

}

void compute_grad_J()

{

double first_term, second_term;

compute_K_t();
compute_K_theta();
K_theta_dot_K_t = Frobenius(K_theta, K_t, chunk_size);
K_theta_dot_K_theta = Frobenius(K_theta, K_theta, chunk_size);
K_t_dot_K_t = Frobenius(K_t, K_t, chunk_size);

for(ind_theta=1; ind_theta<=dim_input; ind_theta++)
  {
  compute_grad_K_theta(ind_theta);
  grad_K_theta_dot_K_t = Frobenius(grad_K_theta, K_t, chunk_size);
  grad_K_theta_dot_K_theta = Frobenius(grad_K_theta, K_theta, chunk_size);
  first_term = grad_K_theta_dot_K_t / sqrt(K_theta_dot_K_theta * K_t_dot_K_t);
  second_term = K_theta_dot_K_t * grad_K_theta_dot_K_theta /
  (K_theta_dot_K_theta * sqrt(K_theta_dot_K_theta * K_t_dot_K_t));
  grad_J_theta[ind_theta] = first_term - second_term;
  }

}

double compute_objective_function()

{

double J_theta=0.0;

compute_K_t();
compute_K_theta();
K_theta_dot_K_t = Frobenius(K_theta, K_t, chunk_size);
K_theta_dot_K_theta = Frobenius(K_theta, K_theta, chunk_size);
K_t_dot_K_t = Frobenius(K_t, K_t, chunk_size);
J_theta = K_theta_dot_K_t / sqrt(K_theta_dot_K_theta * K_t_dot_K_t);

printf("\nValue of the objective function: %lf", J_theta);
return J_theta;

}

void compute_epsilon()

{

norm_theta = 0.0;
norm_grad_J = 0.0;

for(k=1; k<=dim_input; k++)
  {
  norm_theta += theta_2[k];
  norm_grad_J += grad_J_theta[k] * grad_J_theta[k];
  }

norm_theta = sqrt(norm_theta);
norm_grad_J = sqrt(norm_grad_J);

epsilon = norm_theta / (1000.0 * norm_grad_J);
if((iter%step)==0)
  printf("\nValue of epsilon: %e", epsilon);

}

void update_theta()

{

if((iter%step)== 0)
  initial_value = compute_objective_function();

for(ind_theta=1; ind_theta<=dim_input; ind_theta++)
  {
  theta[ind_theta] += epsilon * grad_J_theta[ind_theta];
  theta_2 [ind_theta] = theta[ind_theta] * theta[ind_theta];
  }

if((iter%step)== 0)
  {
  final_value = compute_objective_function();
  ratio = 100.0 * (final_value - initial_value) / initial_value;
  printf("\nRelative increase of the objective function: %e%%\n", ratio);
/* Pause(""); */
  }

/*
if((iter%step)==0)
  {
  printf("\nValue of vector theta:\n\n");

  for(ind_theta=1; ind_theta<=dim_input; ind_theta++)
    printf("%f\n", theta[ind_theta]);
  }
*/

}

void write_theta()

{

if((fc=fopen(fichier_theta, "w"))==NULL)
  {
  printf("\nFile of current vector theta %s: cannot be open...\n", fichier_theta);
  exit(0);
  }

for(i=1; i<=dim_input; i++)
  fprintf(fc, "%le\n", mu * theta_2[i]);

fclose(fc);

}
